"""Methods on YSH Dict"""

from __future__ import print_function

from _devbuild.gen.value_asdl import (value, value_e, value_t, Obj)
from _devbuild.gen.syntax_asdl import loc

from core import error
from core import vm
from frontend import typed_args
from mycpp import mylib, mops
from mycpp.mylib import log, tagswitch
from ysh import val_ops

from typing import cast, List

_ = log


class Keys(vm._Callable):

    def __init__(self):
        # type: () -> None
        pass

    def Call(self, rd):
        # type: (typed_args.Reader) -> value_t

        dictionary = rd.PosDict()
        rd.Done()

        keys = [value.Str(k) for k in dictionary.keys()]  # type: List[value_t]
        return value.List(keys)


class Values(vm._Callable):

    def __init__(self):
        # type: () -> None
        pass

    def Call(self, rd):
        # type: (typed_args.Reader) -> value_t

        dictionary = rd.PosDict()
        rd.Done()

        values = dictionary.values()  # type: List[value_t]
        return value.List(values)


class Erase(vm._Callable):

    def __init__(self):
        # type: () -> None
        pass

    def Call(self, rd):
        # type: (typed_args.Reader) -> value_t

        dictionary = rd.PosDict()
        key = rd.PosStr()
        rd.Done()

        mylib.dict_erase(dictionary, key)
        return value.Null


class Clear(vm._Callable):

    def __init__(self):
        # type: () -> None
        pass

    def Call(self, rd):
        # type: (typed_args.Reader) -> value_t

        dictionary = rd.PosDict()
        rd.Done()

        dictionary.clear()
        return value.Null


class Inc(vm._Callable):

    def __init__(self):
        # type: () -> None
        pass

    def Call(self, rd):
        # type: (typed_args.Reader) -> value_t

        dictionary = rd.PosDict()
        key = rd.PosStr()
        inc_val = rd.OptionalValue()
        rd.Done()

        # If inc_val is provided, we treat it as the definitive type of the
        # value to be incremented, if it isn't, we rely on the existing
        # value in the dictionary (if that doesn't exist, the default is 0)
        if inc_val is None:
            if key in dictionary:
                typed_val = dictionary[key]
            else:
                typed_val = value.Int(mops.BigInt(0))
        else:
            typed_val = inc_val

        with tagswitch(typed_val) as case:
            if case(value_e.Int):
                if inc_val is None:
                    inc_int = 1
                else:
                    inc_int = val_ops.ToInt(inc_val, '', loc.Missing)
                original_int = val_ops.ToInt(
                    dictionary.get(key, value.Int(mops.ZERO)),
                    'Inc() expected int', loc.Missing)
                dictionary[key] = value.Int(mops.BigInt(original_int +
                                                        inc_int))

            elif case(value_e.Float):
                if inc_val is None:
                    inc_float = 1.0
                else:
                    inc_float = val_ops.ToFloat(inc_val, '', loc.Missing)
                original_float = val_ops.ToFloat(
                    dictionary.get(key, value.Float(0.0)),
                    'Inc() expected float', loc.Missing)
                dictionary[key] = value.Float(original_float + inc_float)
            else:
                raise error.TypeErr(typed_val, 'inc() expected Int or float',
                                    rd.BlamePos())

        return value.Null


class Get(vm._Callable):

    def __init__(self):
        # type: () -> None
        pass

    def Call(self, rd):
        # type: (typed_args.Reader) -> value_t

        obj = rd.PosValue()
        key = rd.PosStr()
        default_value = rd.OptionalValue()
        rd.Done()

        UP_obj = obj
        with tagswitch(obj) as case:
            if case(value_e.Dict):
                obj = cast(value.Dict, UP_obj)
                d = obj.d
            elif case(value_e.Obj):
                obj = cast(Obj, UP_obj)
                d = obj.d
            else:
                raise error.TypeErr(obj, 'get() expected Dict or Obj',
                                    rd.BlamePos())

        if default_value is None:
            default_value = value.Null
        return d.get(key, default_value)
